/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: Marshaller.h 12931 2019-04-04 10:06:45Z AlexanderMinor $
   Last Modification:
    $Author: AlexanderMinor $
    $Date: 2019-04-04 12:06:45 +0200 (Do, 04 Apr 2019) $
    $Revision: 12931 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
    Declaration of the cifX API Marshaller class interface
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     3         03.04.2019  AM       Added member m_tDeviceInfo
     2         17.07.2015  RM       Changed header MarshallerFrame.h to MarshallerFrameDLL.h
     1         xx                   created
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file Marshaller.h
/// Declaration of the cifX API Marshaller class interface
/////////////////////////////////////////////////////////////////////////////

#pragma once

#include "APIInterface.h"
#include "cifXErrors.h"
#include "MarshallerFrame.h"

#include "Hil_SystemCmd.h"

#include <string>
#include <map>

#include "TransportLayer.h"
#include "DeviceHandler.h"

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CMarshaller
/// Single instance object which handles the translation of the cifX API into
/// Marshaller objects
///////////////////////////////////////////////////////////////////////////////////////////
class CMarshaller
  : public CDataLayer
{
  CMarshaller();
public:
  CMarshaller(CTransportLayer* pcTransportLayer,
              CEndpoint*       pcEndpoint);

  virtual ~CMarshaller(void);

  virtual bool            ReceiveData(PHIL_TRANSPORT_PACKET ptPacket);
  virtual void            SendComplete(PHIL_TRANSPORT_HEADER ptHeader);

  virtual int32_t         Init(void);
  virtual void            Deinit(void);
  virtual int32_t         EnumDevice(uint32_t ulBoard, uint32_t ulSize, void* pvBoardInfo);
  virtual int32_t         EnumChannel(uint32_t ulBoard, uint32_t ulChannel, uint32_t ulSize, void* pvChannelInfo);
  virtual int32_t         RestartDevice(char* szBoard);

  /* System device depending functions */
  virtual int32_t xSysdeviceOpen              ( CIFXHANDLE  hDriver, char*   szBoard, CIFXHANDLE* phSysdevice);
  virtual int32_t xSysdeviceClose             ( CIFXHANDLE  hSysdevice);
  virtual int32_t xSysdeviceGetMBXState       ( CIFXHANDLE  hSysdevice, uint32_t* pulRecvPktCount, uint32_t* pulSendPktCount); 
  virtual int32_t xSysdevicePutPacket         ( CIFXHANDLE  hSysdevice, CIFX_PACKET* ptSendPkt, uint32_t ulTimeout);
  virtual int32_t xSysdeviceGetPacket         ( CIFXHANDLE  hSysdevice, uint32_t ulSize, CIFX_PACKET* ptRecvPkt, uint32_t ulTimeout);
  virtual int32_t xSysdeviceInfo              ( CIFXHANDLE  hSysdevice, uint32_t ulCmd, uint32_t ulSize, void* pvInfo);

  virtual int32_t xSysdeviceFindFirstFile     ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xSysdeviceFindNextFile      ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xSysdeviceDownload          ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t ulFileSize, 
                                                                    PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xSysdeviceUpload            ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t* pulFileSize, 
                                                                    PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);

  virtual int32_t xSysdeviceReset             ( CIFXHANDLE  hSysdevice, uint32_t ulTimeout);
  virtual int32_t xSysdeviceResetEx           ( CIFXHANDLE  hSysdevice, uint32_t ulTimeout, uint32_t ulMode);

  /* Channel depending functions */
  virtual int32_t xChannelOpen                ( CIFXHANDLE  hDriver,  char* szBoard, uint32_t ulChannel, CIFXHANDLE* phChannel);
  virtual int32_t xChannelClose               ( CIFXHANDLE  hChannel);
  virtual int32_t xChannelFindFirstFile       ( CIFXHANDLE  hChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xChannelFindNextFile        ( CIFXHANDLE  hChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);

  virtual int32_t xChannelDownload            ( CIFXHANDLE  hChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t ulFileSize, 
                                                                  PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xChannelUpload              ( CIFXHANDLE  hChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t* pulFileSize,
                                                                  PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);

  virtual int32_t xChannelGetMBXState         ( CIFXHANDLE  hChannel, uint32_t* pulRecvPktCount, uint32_t* pulSendPktCount);
  virtual int32_t xChannelPutPacket           ( CIFXHANDLE  hChannel, CIFX_PACKET*  ptSendPkt, uint32_t ulTimeout);
  virtual int32_t xChannelGetPacket           ( CIFXHANDLE  hChannel, uint32_t ulSize, CIFX_PACKET* ptRecvPkt, uint32_t ulTimeout);
  virtual int32_t xChannelGetSendPacket       ( CIFXHANDLE  hChannel, uint32_t ulSize, CIFX_PACKET* ptRecvPkt);

  virtual int32_t xChannelConfigLock          ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);
  virtual int32_t xChannelReset               ( CIFXHANDLE  hChannel, uint32_t ulResetMode, uint32_t ulTimeout);
  virtual int32_t xChannelInfo                ( CIFXHANDLE  hChannel, uint32_t ulSize, void* pvChannelInfo);
  virtual int32_t xChannelWatchdog            ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulTrigger);
  virtual int32_t xChannelHostState           ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);
  virtual int32_t xChannelBusState            ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);

  virtual int32_t xChannelIOInfo              ( CIFXHANDLE  hChannel, uint32_t ulCmd,        uint32_t ulAreaNumber, uint32_t ulSize, void* pvData);
  virtual int32_t xChannelIORead              ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulOffset,     uint32_t ulDataLen, void* pvData, uint32_t ulTimeout);
  virtual int32_t xChannelIOWrite             ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulOffset,     uint32_t ulDataLen, void* pvData, uint32_t ulTimeout); 
  virtual int32_t xChannelIOReadSendData      ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulOffset,     uint32_t ulDataLen, void* pvData);

  virtual int32_t xChannelControlBlock        ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);
  virtual int32_t xChannelCommonStatusBlock   ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);
  virtual int32_t xChannelExtendedStatusBlock ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);
  virtual int32_t xChannelUserBlock           ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);

  virtual int32_t xChannelPLCMemoryPtr        ( CIFXHANDLE  hChannel, uint32_t ulCmd ,        void* pvMemoryInfo)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulCmd);
    UNREFERENCED_PARAMETER(pvMemoryInfo);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCIsReadReady      ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t* pulReadState)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);
    UNREFERENCED_PARAMETER(pulReadState);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCIsWriteReady     ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t* pulWriteState)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);
    UNREFERENCED_PARAMETER(pulWriteState);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCActivateWrite    ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCActivateRead     ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }

protected:

  typedef struct MARSH_CHANNEL_INFO_Ttag
  {
    LONG  lRefCount;
    DWORD ulChannelNr;    /*!< 0xFFFFFFFF == HIL_SYSTEM_CHANNEL === system device */
    DWORD dwRemoteHandle;

  } MARSH_CHANNEL_INFO_T, *PMARSH_CHANNEL_INFO_T;

  typedef std::map<DWORD, PMARSH_CHANNEL_INFO_T> MARSH_CHANNEL_MAP;
  typedef std::map<std::string, MARSH_CHANNEL_MAP> MARSH_DEVICE_MAP;

  CRITICAL_SECTION            m_tDeviceMapLock;
  CRITICAL_SECTION            m_tcsResetLock;
  
  /* Map of the managed devices */
  MARSH_DEVICE_MAP            m_cmDevice;


  CEndpoint*                  m_pcEndpoint;

  DWORD                       m_dwClassFactory;       /*!< Class factory handle of device communication */
  DWORD                       m_dwDriverHandle;       /*!< Driver handel of device communication */

  WORD                        m_usSequenceNr;
  bool                        m_fSequenceNrSupport;
  DWORD                       m_ulVersion;

  HANDLE                      m_hReConnectEvent;    /*!< Event, which should be signaled on reconnect */
  HIL_HW_IDENTIFY_CNF_DATA_T  m_tDeviceInfo;        /*!< Hardware configuration data */

  int32_t    CreateObject(DWORD dwDestHandle, DWORD dwObjectType, DWORD& dwObjectHandle);
  int32_t    DeleteDeviceObject(DWORD dwObjectHandle, DWORD dwChannelNr);

  int32_t    QueryMarshallerVersion(void);
  int32_t    OpenDriver(void);
  int32_t    CloseDriver(void);

  int32_t    MethodeInfo( PMARSH_CHANNEL_INFO_T ptDevice, uint32_t ulCmd, uint32_t ulSize, void* pvInfo, uint32_t ulType);
  int32_t    Reset( PMARSH_CHANNEL_INFO_T ptDevice, uint32_t ulResetMode, uint32_t ulType, uint32_t ulTimeout);
  int32_t    OpenChannel(std::string szDevice, DWORD dwChannelNr, CIFXHANDLE* phChannel);
  int32_t    CloseChannel(PMARSH_CHANNEL_INFO_T ptChannel, bool fSendRequest = true, bool fForceRemove = false);
  int32_t    PutPacket(PMARSH_CHANNEL_INFO_T ptChannel, CIFX_PACKET* ptPacket, uint32_t ulTimeout);
  int32_t    GetPacket(PMARSH_CHANNEL_INFO_T ptChannel, CIFX_PACKET* ptPacket, uint32_t ulPacketLen, uint32_t ulTimeout);
  int32_t    GetMbxState(PMARSH_CHANNEL_INFO_T ptChannel, uint32_t* pulRecvCnt, uint32_t* pulSendCnt);
  int32_t    ReadHardwareIdentify(BOARD_INFORMATION& tBoardInfo);
  

  void       GenerateHeader(PMARSHALLER_DATA_FRAME_HEADER_T ptHeader,
                            uint32_t ulHandle, uint32_t ulMethodId, 
                            bool fRequest, uint32_t ulDataSize);

  int32_t    VerifyTransferResult (PMARSHALLER_DATA_FRAME_HEADER_T ptReq, 
                                   PMARSHALLER_DATA_FRAME_HEADER_T ptCnf);

  int32_t    PrepareReconnect( void);
  int32_t    ReConnectToDevice( void);

  int32_t    CloseConnectionsToDevice( PMARSH_CHANNEL_INFO_T ptChannel);

  int32_t    ChannelReadWriteBlock(PMARSH_CHANNEL_INFO_T ptChannel, uint32_t ulCmd, 
                                   uint32_t ulOffset, uint32_t ulDataLen, void* pvData,
                                   uint32_t ulMethodID);
};
